/*
 * Copyright (c) 2018, 2021, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package jdk.jpackage.internal;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import jdk.jpackage.internal.Arguments.CLIOptions;
import static jdk.jpackage.internal.StandardBundlerParam.LAUNCHER_DATA;
import static jdk.jpackage.internal.StandardBundlerParam.APP_NAME;
import static jdk.jpackage.internal.StandardBundlerParam.MENU_HINT;
import static jdk.jpackage.internal.StandardBundlerParam.SHORTCUT_HINT;

/*
 * AddLauncherArguments
 *
 * Processes a add-launcher properties file to create the Map of
 * bundle params applicable to the add-launcher:
 *
 * BundlerParams p = (new AddLauncherArguments(file)).getLauncherMap();
 *
 * A add-launcher is another executable program generated by either the
 * create-app-image mode or the create-installer mode.
 * The add-launcher may be the same program with different configuration,
 * or a completely different program created from the same files.
 *
 * There may be multiple add-launchers, each created by using the
 * command line arg "--add-launcher <file path>
 *
 * The add-launcher properties file may have any of:
 *
 * appVersion
 * module
 * main-jar
 * main-class
 * icon
 * arguments
 * java-options
 * win-console
 * win-shortcut
 * win-menu
 * linux-app-category
 * linux-shortcut
 *
 */
class AddLauncherArguments {

    private final String name;
    private final String filename;
    private Map<String, String> allArgs;
    private Map<String, ? super Object> bundleParams;

    AddLauncherArguments(String name, String filename) {
        this.name = name;
        this.filename = filename;
    }

    private void initLauncherMap() {
        if (bundleParams != null) {
            return;
        }

        allArgs = Arguments.getPropertiesFromFile(filename);
        allArgs.put(CLIOptions.NAME.getId(), name);

        bundleParams = new HashMap<>();
        String mainJar = getOptionValue(CLIOptions.MAIN_JAR);
        String mainClass = getOptionValue(CLIOptions.APPCLASS);
        String module = getOptionValue(CLIOptions.MODULE);

        if (module != null && mainClass != null) {
            Arguments.putUnlessNull(bundleParams, CLIOptions.MODULE.getId(),
                    module + "/" + mainClass);
        } else if (module != null) {
            Arguments.putUnlessNull(bundleParams, CLIOptions.MODULE.getId(),
                    module);
        } else {
            Arguments.putUnlessNull(bundleParams, CLIOptions.MAIN_JAR.getId(),
                    mainJar);
            Arguments.putUnlessNull(bundleParams, CLIOptions.APPCLASS.getId(),
                    mainClass);
        }

        Arguments.putUnlessNull(bundleParams, CLIOptions.NAME.getId(),
                getOptionValue(CLIOptions.NAME));

        Arguments.putUnlessNull(bundleParams, CLIOptions.VERSION.getId(),
                getOptionValue(CLIOptions.VERSION));

        Arguments.putUnlessNull(bundleParams, CLIOptions.RELEASE.getId(),
                getOptionValue(CLIOptions.RELEASE));

        String value = getOptionValue(CLIOptions.ICON);
        Arguments.putUnlessNull(bundleParams, CLIOptions.ICON.getId(),
                (value == null) ? null : Path.of(value));

        if (Platform.isWindows())  {
            Arguments.putUnlessNull(bundleParams,
                    CLIOptions.WIN_CONSOLE_HINT.getId(),
                    getOptionValue(CLIOptions.WIN_CONSOLE_HINT));
            Arguments.putUnlessNull(bundleParams, SHORTCUT_HINT.getID(),
                    getOptionValue(CLIOptions.WIN_SHORTCUT_HINT));
            Arguments.putUnlessNull(bundleParams, MENU_HINT.getID(),
                    getOptionValue(CLIOptions.WIN_MENU_HINT));
        }

        if (Platform.isLinux())  {
            Arguments.putUnlessNull(bundleParams, CLIOptions.LINUX_CATEGORY.getId(),
                    getOptionValue(CLIOptions.LINUX_CATEGORY));
            Arguments.putUnlessNull(bundleParams, SHORTCUT_HINT.getID(),
                    getOptionValue(CLIOptions.LINUX_SHORTCUT_HINT));
        }

        // "arguments" and "java-options" even if value is null:
        if (allArgs.containsKey(CLIOptions.ARGUMENTS.getId())) {
            String argumentStr = getOptionValue(CLIOptions.ARGUMENTS);
            bundleParams.put(CLIOptions.ARGUMENTS.getId(),
                    Arguments.getArgumentList(argumentStr));
        }

        if (allArgs.containsKey(CLIOptions.JAVA_OPTIONS.getId())) {
            String jvmargsStr = getOptionValue(CLIOptions.JAVA_OPTIONS);
            bundleParams.put(CLIOptions.JAVA_OPTIONS.getId(),
                    Arguments.getArgumentList(jvmargsStr));
        }
    }

    private String getOptionValue(CLIOptions option) {
        if (option == null || allArgs == null) {
            return null;
        }

        String id = option.getId();

        if (allArgs.containsKey(id)) {
            return allArgs.get(id);
        }

        return null;
    }

    Map<String, ? super Object> getLauncherMap() {
        initLauncherMap();
        return bundleParams;
    }

    static Map<String, ? super Object> merge(
            Map<String, ? super Object> original,
            Map<String, ? super Object> additional, String... exclude) {
        Map<String, ? super Object> tmp = new HashMap<>(original);
        List.of(exclude).forEach(tmp::remove);

        // remove LauncherData from map so it will be re-computed
        tmp.remove(LAUNCHER_DATA.getID());
        // remove "application-name" so it will be re-computed
        tmp.remove(APP_NAME.getID());

        if (additional.containsKey(CLIOptions.MODULE.getId())) {
            tmp.remove(CLIOptions.MAIN_JAR.getId());
            tmp.remove(CLIOptions.APPCLASS.getId());
        } else if (additional.containsKey(CLIOptions.MAIN_JAR.getId())) {
            tmp.remove(CLIOptions.MODULE.getId());
        }
        if (additional.containsKey(CLIOptions.ARGUMENTS.getId())) {
            // if add launcher properties file contains "arguments", even with
            // null value, disregard the "arguments" from command line
            tmp.remove(CLIOptions.ARGUMENTS.getId());
        }
        if (additional.containsKey(CLIOptions.JAVA_OPTIONS.getId())) {
            // same thing for java-options
            tmp.remove(CLIOptions.JAVA_OPTIONS.getId());
        }
        tmp.putAll(additional);
        return tmp;
    }

}
